k6 v0.42.0 is here!  :tada:

This release includes:
- A tiny breaking change to improve WebSockets response handling.
- A new experimental output.
- More features in our experimental WebSocket module.
- Wildcard support for `hosts`.
- Some bug fixes, UX improvements, and maintenance.

## Breaking changes

- [#2712](https://github.com/grafana/k6/pull/2712) `k6/ws` returns an HTTP response for failed connections instead of an undefined behavior. Thanks, @brietaylor.

## New Features

### Experimental Prometheus Remote Write Output [#2784](https://github.com/grafana/k6/pull/2784)

This release brings a new builtin Output to any [Prometheus Remote Write](https://docs.google.com/document/d/1LPhVRSFkGNSuU1fBd81ulhsCPR4hkSZyyBj1SZ8fWOM/edit) implementations (e.g. [Prometheus](https://prometheus.io/docs/prometheus/latest/feature_flags/#remote-write-receiver), [Mimir](https://grafana.com/docs/mimir/latest/operators-guide/reference-http-api/#remote-write)). This is an experimental feature, and future releases could introduce breaking changes.

The following example uses `k6 run` with the new output. It uses the defaults options, such as the Remote Write server URL (http://localhost:9090/api/v1/write):

```sh
k6 run -o experimental-prometheus-rw script.js
```

It supports the new and convenient experimental [Native Histogram](https://prometheus.io/docs/practices/histograms) feature, added in Prometheus `v2.40.0`. It's not enabled by default, but we expect to make it the default way to map k6 [Trend metrics](https://k6.io/docs/javascript-api/k6-metrics/trend) once the Prometheus project signals that its mature enough and when more Remote Write implementations support it. For now, if you want to use it, you need to set the environment variable `K6_PROMETHEUS_RW_TREND_AS_NATIVE_HISTOGRAM` to `true`.

You can find [complete documentation](https://k6.io/docs/results-output/real-time/prometheus-remote-write) with more examples, use cases, and available configurations.

### More features for the experimental websockets module [#2786](https://github.com/grafana/k6/pull/2786) 

The `k6/experimental/websockets` module that we announced in the [v0.40.0 release](https://github.com/grafana/k6/releases/tag/v0.40.0) got an update that extends its functionality.

It brings some useful features that the `k6/ws` module already has, like cookies, custom headers, compression and tags customization support, the syntax to define event handlers (`onopen`, `onmessage`, etc.) and `ping`/`pong` functionality.

This is still an experimental module, but with the recent changes we think it's usable for most users. So whether you're writing a new WebSocket test, or currently using the `k6/ws` module, we invite you to give it a try, and report any issues in the project's [issue tracker](https://github.com/grafana/xk6-websockets/issues).

<details>
<summary> Expand to see an example of the new WebSockets functionality</summary>

This example customizes tags for a WebSocket connection, sets up handlers using the new `on*` syntax, and demonstrates the `ping`/`pong` feature.

```javascript
import { WebSocket } from "k6/experimental/websockets";
import {
  setTimeout,
  clearTimeout,
  setInterval,
  clearInterval
} from "k6/experimental/timers";

const CLOSED_STATE = 3

export default function () {
  const params = {
    "tags": {
      "my_tag": "hello"
    }
  };

  const ws = new WebSocket('ws://localhost:10000', null, params);

  ws.onopen = () => {
    console.log('connected');
    ws.send(Date.now().toString());
  };

  let intervalId = setInterval(() => {
    ws.ping();
    console.log("Pinging every 1 sec (setInterval test)");
  }, 1000);

  let timeout1id = setTimeout(function () {
    console.log('2 seconds passed, closing the socket');
    clearInterval(intervalId);
    ws.close();
  }, 2000);

  ws.onclose = () => {
    clearTimeout(timeout1id);
    console.log('disconnected');
  };

  ws.onping = () => {
    console.log("PING!");
  };

  ws.onpong = () => {
    console.log("PONG!");
  };

  // Multiple event handlers on the same event
  ws.addEventListener("pong", () => {
    console.log("OTHER PONG!");
  });

  ws.onmessage = (m) => {
    let parsed = parseInt(m.data, 10)
    if (Number.isNaN(parsed)) {
      console.log('Not a number received: ', m.data);
      return
    }

    console.log(`Roundtrip time: ${Date.now() - parsed} ms`);

    let timeoutId = setTimeout(function () {
      if (ws.readyState == CLOSED_STATE) {
        console.log("Socket closed, not sending anything");

        clearTimeout(timeoutId);
        return;
      }

      ws.send(Date.now().toString());
    }, 500);
  };

  ws.onerror = (e) => {
    if (e.error != "websocket: close sent") {
      console.log('An unexpected error occurred: ', e.error);
    }
  };
};
```
</details>

The [module docs](https://k6.io/docs/javascript-api/k6-experimental/websockets) has a complete reference, and some examples.

### Wildcard support for `hosts` option [#2747](https://github.com/grafana/k6/pull/2747)

Thanks to the great effort from @eugercek, the [hosts](https://k6.io/docs/using-k6/k6-options/reference/#hosts) option now accepts domains that contain a wildcard at the beginning.
It can be helpful for setting multiple subdomains of the same domain, so instead of setting `subdomain1.k6.io': '1.2.3.4', 'subdomain2.k6.io': '1.2.3.4'` it is possible to use the wildcard for setting directly `*.k6.io: '1.2.3.4'`.

```js
export const options = {
  hosts: {
    '*.k6.io': '1.2.3.4',
  },
}
```

### Enhancements and UX improvements

- [#2660](https://github.com/grafana/k6/pull/2660) Pre-loads the operating system TLS certificates. Thanks, @tbourrely.
- [#2791](https://github.com/grafana/k6/pull/2791) Initializes VUs for `setup()` and `teardown()` only if they are defined in the script.

## Bug fixes

- [#2759](https://github.com/grafana/k6/pull/2759) Ensures the evaluation of thresholds over trend metrics' median.
- [#2759](https://github.com/grafana/k6/pull/2789) Fixes a few potential Output data races for interrupted test runs.
- [#2767](https://github.com/grafana/k6/pull/2767) Fixes the emission of `ws_session_duration` when setup throws an error.
- [#2773](https://github.com/grafana/k6/pull/2773) Ensures that JavaScript runtime makes only one copy of the exports for each module including built-in ones.

## Maintenance and internal improvements

We had a few minor changes in this release:

- [#2757](https://github.com/grafana/k6/pull/2757) goja runtime has been updated.
- [#2768](https://github.com/grafana/k6/pull/2768) `WS.Connect()` has been refactored.
- [#2770](https://github.com/grafana/k6/pull/2770) Refactored parts of the js module.
- [#2782](https://github.com/grafana/k6/pull/2782) Covered more relative path test cases for `require` and `open`.
- [#2789](https://github.com/grafana/k6/pull/2789), [#2792](https://github.com/grafana/k6/pull/2792), [#2795](https://github.com/grafana/k6/pull/2795), [#2796](https://github.com/grafana/k6/pull/2796) Improved stability of the integration tests.
- [#2791](https://github.com/grafana/k6/pull/2777) Optimized the performance of the internal Trie implementation. Thanks, @eugercek.

**Full Changelog**: https://github.com/grafana/k6/compare/v0.41.0...v0.42.0
